/**
* \file: automounter_api.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#define _GNU_SOURCE

#include <sys/syscall.h>
#include <unistd.h>
#include <pthread.h>
#include <assert.h>

#include "automounter_api.h"
#include "automounter_api_info.h"
#include "automounter_api_events.h"
#include "automounter_api_ctrl.h"

#include "am_api_fsm.h"
#include "am_api_mutex.h"
#include "am_api_dispatcher.h"
#include "am_api_events.h"
#include "am_api_control.h"

#include "utils/version_info.h"
#include "utils/logger.h"
#include "utils/automounter_types_internal.h"

//------------------------------------------ private function declaration --------------------------------------------
static int request_id_cntr=0;
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------------ private function declaration --------------------------------------------
static int automounter_api_get_new_request_id(void);
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------------ public function definitions ---------------------------------------------
error_code_t automounter_api_init(const char *app_identifer, logger_loglevel_t loglevel, bool console_log_enabled)
{
	error_code_t result;

	//we need to wait for a thread that is currently calling callbacks of the application, since this init call might
	//reinitialize the receiver buffer, which contains data that applications might use at this time.
	if (am_api_mutex_entering_event_dispatcher()==RESULT_INVALID)
	{
		char thread_name[256];
		int thread_id;
		thread_id=syscall(SYS_gettid);
		pthread_getname_np(pthread_self(),thread_name, sizeof(thread_name));
		logger_log_error("AUTOMOUNTER_API - The function automounter_api_init() is called out of an"
						" callback function registered with the automounter API. This is not allowed!!!"
						" See the design document and the API description for further details. This error is not recoverable,"
						" we are exiting now. (Calling thread: Id=%ld, Name=%s)", thread_id, thread_name);
		assert(false);
	}
	am_api_mutex_entering_library();

	request_id_cntr=0;
	result=am_api_fsm_signal_init_request(app_identifer,loglevel,console_log_enabled);

	am_api_mutex_leaving_library();
	am_api_mutex_leaving_event_dispatcher();
	return result;
}

void automounter_api_deinit(void)
{
	//we need to wait for a thread that is currently calling callbacks of the application, since this deinit call will
	//destroy the receiver buffer, which contains data that applications might use at this time.
	if (am_api_mutex_entering_event_dispatcher()==RESULT_INVALID)
	{
		char thread_name[256];
		int thread_id;
		thread_id=syscall(SYS_gettid);
		pthread_getname_np(pthread_self(),thread_name, sizeof(thread_name));
		logger_log_error("AUTOMOUNTER_API - The function automounter_api_deinit() is called out of an"
						" callback function registered with the automounter API. This is not allowed!!!"
						" See the design document and the API description for further details. This error is not recoverable,"
						" we are exiting now. (Calling thread: Id=%ld, Name=%s)", thread_id, thread_name);
		assert(false);
	}
	am_api_mutex_entering_library();
	am_api_fsm_signal_deinit_request();
	am_api_mutex_leaving_library();
	am_api_mutex_leaving_event_dispatcher();
}

automounter_api_state_t automounter_api_get_state(void)
{
	//--- no mutex needed in here
	return am_api_fsm_get_state();
}

error_code_t automounter_api_try_connect(void)
{
	error_code_t result;

	am_api_mutex_entering_library();
	result=am_api_fsm_signal_try_connect_request();
	am_api_mutex_leaving_library();

	return result;
}


error_code_t automounter_api_connect(void)
{
	error_code_t result;

	am_api_mutex_entering_library();
	result=am_api_fsm_signal_connect_request();
	am_api_mutex_leaving_library();

	return result;
}


void automounter_api_disconnect(void)
{
	am_api_mutex_entering_library();
	am_api_fsm_signal_disconnect_request();
	am_api_mutex_leaving_library();
}

const char *automounter_api_get_version_string(void)
{
	//--- no mutex needed in here
	return VERSION_INFO_FORMATED_LINE;
}

int automounter_api_get_pollfd(void)
{
	//--- no mutex needed in here
	return am_api_dispatcher_get_pollfd();
}

void automounter_api_dispatch_event(void)
{
	if (am_api_mutex_entering_event_dispatcher()==RESULT_INVALID)
	{
		char thread_name[256];
		int thread_id;
		thread_id=syscall(SYS_gettid);
		pthread_getname_np(pthread_self(),thread_name, sizeof(thread_name));
		logger_log_error("AUTOMOUNTER_API - The function automounter_api_dispatch_event() is called out of an"
				" callback function registered with the automounter API. This is not allowed!!!"
				" See the design document and the API description for further details. For now, the call of "
				" automounter_api_dispatch_event() is ignored. (Calling thread: Id=%ld, Name=%s)", thread_id, thread_name);
		// we dispatcher mutex is not locked when the function am_api_mutex_entering_event_dispatcher()
		// returns RESULT_INVALID. So we do not need to unlock anything here.
		return;
	}

	am_api_mutex_entering_library();

	// The dispatch function internally ensures that the mutex is released before any external callback is called and
	// locked again as soon as the callback returns. This enables the way to call API functions of the library out
	// of a callback function.
	// The dispatcher mutex is not released to ensure that the dispatch functions is only called ones at a time.
	// This is needed to protect the read buffer, in which the data is stored that is passed to the callback functions.
	if (am_api_fsm_get_state()!=AM_API_UNINITIALIZED)
		am_api_dispatcher_dispatch_event();

	am_api_mutex_leaving_library();
	am_api_mutex_leaving_event_dispatcher();
}

void automounter_api_register_callbacks(const automounter_api_callbacks_t *callback_funcs)
{
	am_api_mutex_entering_library();
	if (am_api_fsm_get_state()!=AM_API_UNINITIALIZED)
		am_api_events_register_callbacks(callback_funcs);
	am_api_mutex_leaving_library();
}

error_code_t automounter_api_get_snapshot(snapshot_scope_t scope, int *request_id_ptr)
{
	error_code_t result;
	int request_id;

	am_api_mutex_entering_library();
	if (am_api_fsm_get_state()==AM_API_INITIALIZED_CONNECTED)
	{
		request_id=automounter_api_get_new_request_id();
		if (request_id_ptr!=NULL)
			*request_id_ptr=request_id;
		result=am_api_control_get_snapshot(scope,request_id);
	}
	else
	{
		logger_log_error("AUTOUMOUNTER_API - Invalid umount request. The shared library is not connected with the"
				" automounter daemon.");
		result=RESULT_INVALID;
	}
	am_api_mutex_leaving_library();

	return result;
}

error_code_t automounter_api_umount_device(const char *device_id, int request_id,
		request_done_callback_t done_callback_func)
{
	error_code_t result;

	am_api_mutex_entering_library();
	if (am_api_fsm_get_state()==AM_API_INITIALIZED_CONNECTED)
	{
		result=am_api_control_umount_device(device_id,request_id,done_callback_func);
	}
	else
	{
		logger_log_error("AUTOUMOUNTER_API - Invalid umount request. The shared library is not connected with the"
				" automounter daemon.");
		result=RESULT_INVALID;
	}
	am_api_mutex_leaving_library();

	return result;
}

error_code_t automounter_api_umount_partition_by_id(const char *partition_id, int request_id,
		request_done_callback_t done_callback_func)
{
	error_code_t result;

	am_api_mutex_entering_library();
	if (am_api_fsm_get_state()==AM_API_INITIALIZED_CONNECTED)
	{
		result=am_api_control_umount_partition_by_id(partition_id,request_id,done_callback_func);
	}
	else
	{
		logger_log_error("AUTOUMOUNTER_API - Invalid umount request. The shared library is not connected with the"
				" automounter daemon.");
		result=RESULT_INVALID;
	}
	am_api_mutex_leaving_library();

	return result;
}

error_code_t automounter_api_umount_partition_by_mountpoint(const char *mount_point, int request_id,
		request_done_callback_t done_callback_func)
{
	error_code_t result;

	am_api_mutex_entering_library();
	if (am_api_fsm_get_state()==AM_API_INITIALIZED_CONNECTED)
	{
		result=am_api_control_umount_partition_by_mountpoint(mount_point,request_id,done_callback_func);
	}
	else
	{
		logger_log_error("AUTOUMOUNTER_API - Invalid umount request. The shared library is not connected with the"
				" automounter daemon.");
		result=RESULT_INVALID;
	}
	am_api_mutex_leaving_library();

	return result;
}

error_code_t automounter_api_remount_partition_by_mountpoint(const char *mount_point,const char *options,
		int request_id, request_done_callback_t done_callback_func)
{
	error_code_t result;

	am_api_mutex_entering_library();
	if (am_api_fsm_get_state()==AM_API_INITIALIZED_CONNECTED)
	{
		result=am_api_control_remount_partition_by_mountpoint(mount_point,options,request_id,done_callback_func);
	}
	else
	{
		logger_log_error("AUTOUMOUNTER_API - Invalid remount request. The shared library is not connected with the"
				" automounter daemon.");
		result=RESULT_INVALID;
	}
	am_api_mutex_leaving_library();

	return result;
}

error_code_t automounter_api_remount_partition_by_id(const char *partition_id,const char *options,
		int request_id, request_done_callback_t done_callback_func)
{
	error_code_t result;

	am_api_mutex_entering_library();
	if (am_api_fsm_get_state()==AM_API_INITIALIZED_CONNECTED)
	{
		result=am_api_control_remount_partition_by_id(partition_id,options,request_id,done_callback_func);
	}
	else
	{
		logger_log_error("AUTOUMOUNTER_API - Invalid remount request. The shared library is not connected with the"
				" automounter daemon.");
		result=RESULT_INVALID;
	}
	am_api_mutex_leaving_library();

	return result;
}

const char *automounter_api_get_device_state_string(device_state_t state)
{
	//--- no mutex needed in here
	return automounter_get_device_state_string(state);
}

const char *automounter_api_get_partition_state_string(partition_state_t state)
{
	//--- no mutex needed in here
	return automounter_get_partition_state_string(state);
}

const char *automounter_api_get_partition_unsupported_reason_string(partition_unsupported_reason_t reason)
{
	//--- no mutex needed in here
	return automounter_get_unsupported_reason_string(reason);
}
//--------------------------------------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------------------------------------
static int automounter_api_get_new_request_id(void)
{
	return request_id_cntr++;
}
//--------------------------------------------------------------------------------------------------------------------
